
; Emergency stop signalling. Flashes vehicle brake or hazard lights if deceleration is over 6m/s/s
; switches off flashing if below 2.5m/s/s
; Flash rate 3.85Hz

; Uses 3-axis analog accelerometer to detect braking rate. 
; Compensation included for up or down hill slope using Z output
; X or Y output used for deceleration detection
; during calibration only monitors X or Y output so can use gravity as a reference deceleration source
; X or Y output selection is with a shorting plug for one or the other (JP1). Which selection (X or Y) depends on printed circuit board orientation in vehicle when installed
; It allows a 90 degree orientation option alternative for installation in the vehicle

; calibrations. 
; 1. read and store horizontal position quiescent value of the X or Y output and  the Z output (JP2 in)
; 2. compensation for hill (slope up and down) gain adjustment with VR3  (JP3 in UP/DN position)
; 3. calibrate upper and lower thresholds (VR3 and VR4) (JP3 in Calibrate position)

; Relays: RELAY2 switches on and off at 3.85Hz during emergency braking period. The other relay (RELAY1) is switched on during emergency braking period
; relays are to cause  flashing brake lights or hazard lights. Relay1 and Relay2 required for hazards wiring. Relay2 only for brake lights.

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1

; code protection
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0  & _DEBUG_OFF & _WRT_PROTECT_ALL & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO
;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

;  EEPROM storage

EEPROM0		equ	H'00'	;  X or Y horizontal quiescent value
EEPROM1		equ	H'01'	;  Z horizontal quiescent value 
EEPROM2		equ H'02'	; compensation multiplier 

; bank 0 RAM
CONVERSION	equ	H'20'	; A/D conversion
STORE1			equ	H'21'	; delay register
STORE2			equ	H'22'	; delay register
STORE3			equ	H'23'	; extend delay
DIVIDE			equ	H'24'	; interrupt rate divide by 2
BRAKE			equ	H'25'	; brake flag
UPPER_TRS		equ	H'26'	; upper deceleratioin threshold 
LOWER_TRS	equ	H'27'	; lower deceleration threshold
DECELERATE	equ	H'28'	; deceleration rate
XY_HORIZ		equ	H'29'	; X or Y horizontal quiescent value
Z_HORIZ		equ	H'2A'	; Z horizontal quiescent value 
MULTIPLY		equ	H'2B'	; compensation multiplier 
DIVISOR			equ	H'2C'	; divisor value
MORE_LESS	equ	H'2D'	; increase or decrease flag	
XY_RUN			equ	H'2E'	; current x or y value
Z_RUN			equ	H'2F'	; current z value
PORTA_STO		equ	H'30'	; register copy of PORTA
LOGGER		equ	H'31'	; logger counter for EEPROM
EXTRA_DLY		equ	H'32'	; extra delay extension	

; decelerate values for summation and averaging over 8 readings
DECEL1		equ	H'58'	; decelerate 1
DECEL2		equ	H'59'	; decelerate 2
DECEL3		equ	H'5A'	; decelerate 3
DECEL4		equ	H'5B'	; decelerate 4
DECEL5		equ	H'5C'	; decelerate 5
DECEL6		equ	H'5D'	; decelerate 6
DECEL7		equ	H'5E'	; decelerate 7
DECEL8		equ	H'5F'	; decelerate 8
ADDN0			equ	H'60'	; decelerate addition ms byte	
ADDN1			equ	H'61'	; decelerate addition ls byte	

; math routines
TEMPB1      		equ H'62'
TEMPB0     		equ H'63'
TEMPD       		equ H'64'   	 ; temporary storage
AARGB3		equ	H'65'
AARGB2      		equ H'66'
AARGB1     		equ H'67'
AARGB0      		equ H'68'	; most significant byte of argument A    
BARGB1      		equ H'69'
BARGB0      		equ H'6A'	; most significant byte of argument B  
REMB1       		equ H'6B'
REMB0       		equ H'6C'   	; most significant byte of remainder
LOOPCOUNT   	equ H'6D'   	; loop counter
STO1	    		equ	H'6E'	; AARGB1 storage

; All Banks RAM

W_TMP			equ	H'70'	; storage of w before interrupt
STATUS_TMP	equ	H'71'	; status storage before interrupt

	ORG     H'2100'
; preset  EEPROM locations 

	DE	H'7F'	; XY axis quiescent 2.5V
	DE  H'CC' 	; Z axis quiescent 4V
	DE  D'92' 	; H5C multiplier 	

; ******************************************************************

; start at memory 0

	org		0			; reset vector
	goto	MAIN		; byte 1

	nop
	nop
	nop	

	org    	 4			; interrupt vector
 
INTERRUPT

	bcf		INTCON,TMR0IF	; clear TMRO interrupt flag

; divide by 2
	incf		DIVIDE,f
	btfsc	DIVIDE,0
	retfie					; return if set

; store w and status registers

	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	bcf		STATUS,RP0	; select bank 0
	bcf		STATUS,RP1	; select bank 0

;TIMER0 timeout. 1us x 256 x 256 = 65ms so every 2nd timeout is 130ms half period for 3.85Hz

; remark out as appropriate
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; test
; where relays are kept off. LED off version
;	movlw	B'00000010' 	

; Normal 
; Relay1 on, Relay2 off and  LED off version
	movlw	B'10000010'		; LED1, Q2 off, Q1 on
;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

	btfsc	PORTA_STO,1	; if set then clear. if clear then set

; remark out as appropriate
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; test 
; where relays are off,  LED on version
;	movlw	B'00000000' 	

; Normal
; Relays and LED on
	movlw	B'10000001'		; LED1, Q1 and Q2 on
;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	movwf	PORTA_STO		; store PORTA value
	movwf	PORTA

; restore status and w
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

;******************************************************************************************* 
MAIN

	clrf		PORTB			; all low
	movlw	B'00000010'		; LED1,  Q1 and Q2 off
	movwf	PORTA_STO		; store PORTA value
	movwf	PORTA
	bsf		STATUS,RP0	; select memory bank 1

; set inputs/outputs
	movlw	B'00000111'		; comparators off
	movwf	CMCON
	movlw	B'11100110'		; port B outputs/ inputs set 
	movwf	TRISB			; port B data direction register
	movlw	B'00101100'		; outputs (0) and inputs (1)
	movwf	TRISA			; port A data direction register
	movlw	B'00000111'		; settings (pullups enabled TMR0/256)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'00011110'		; AN1,2,3,4 analog inputs (AN2, AN3 for REF-, REF+)
	movwf	ANSEL
	movlw	B'00000000'		; left justified, references 0V and 5V
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'01100000' 		; Fosc, channel 1 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01101000'		; for 4MHz
	movwf	OSCCON		; osc
; pwm set
	movlw	H'FF'
	movwf	PR2				; PWM period register
	bcf		STATUS,RP0	; memory bank 0
	movlw	D'128'			; 50% duty, 3.9kHz
	movwf	CCPR1L		; ms byte of PWM
	movlw	B'00000000'
	movwf	T2CON
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'		; set PWM mode
	movwf	CCP1CON		; enable PWM operation

; Initial
	clrf		BRAKE
	clrf		DECEL8
	clrf		DECEL7
	clrf		DECEL6
	clrf		DECEL5
	clrf		DECEL4
	clrf		DECEL3
	clrf		DECEL2
	clrf		DECEL1
	movlw	D'02'			; logging from EEPROM3-EEPROM255
	movwf	LOGGER

	goto	STORED	; bypass clearing EEPROM if logging not required

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; initial logging setup
; clear logging from EEPROM3 to 255
CLR_LOG
	incf		LOGGER,w
	btfsc	STATUS,Z		; if increased past 255 to zero, then stop logging
	goto	END_CLEAR	; end clearing EEPROM
	incf		LOGGER,f		; increase logger address
	bsf 		STATUS,RP1	; select memory bank 2
	movwf 	EEADR			; indirect special function register
	bcf 		STATUS,RP1	; select memory bank 0
	movlw	H'FF'			; cleared
	call		EEWRITE		; store in EEPROM
	goto	CLR_LOG		; continue
END_CLEAR
	movlw	D'02'			; logging from EEPROM3-EEPROM255
	movwf	LOGGER
; end of clear logging
;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

STORED
; get stored horizontal quiescent values 
	movlw	EEPROM0
	call		EEREAD
	movwf	XY_HORIZ		; X or Y horizontal quiescent value
	
	movlw	EEPROM1
	call		EEREAD
	movwf	Z_HORIZ		; Z horizontal quiescent value 

; get stored compensation multiplier

	movlw	EEPROM2
	call		EEREAD
	movwf	MULTIPLY		; compensation multiplier

; light LED momentarily
; LED on
	movlw	B'00000000'		; LED1 on
	movwf	PORTA

; delay
	movlw	D'4'
	movwf	EXTRA_DLY
EXTRA
	movlw	D'200'			; delay 1s (5ms per value)
	movwf	STORE3
DELAYX1
	call		DELAYms
	decfsz	STORE3,f
	goto	DELAYX1
; LED off
	movlw	B'00000010'		; LED1 off
	movwf	PORTA

	decfsz	EXTRA_DLY,f
	goto	EXTRA			; extra delay

; >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; read VR3 and VR4 (upper and lower deceleration thresholds )
UL ; upper and lower thresholds
;AN6 VR3 UPPER
; bits 5 4 3
; ch6  1 1 0
	bcf		ADCON0,3
	bsf		ADCON0,4
	bsf		ADCON0,5

; Channel 6 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	UPPER_TRS

; AN5 VR4 LOWER
; bits 5 4 3
; ch3  1 0 1
	bsf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5

; Channel 5 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	LOWER_TRS

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

NORMAL ; (normal running) 
; check jumper links
	btfss	PORTB,5
	goto	QUIESCENT			; quiescient outputs from X or Y and Z when accelerometer in a horizontal position
	btfss	PORTB,1
	goto	UP_DN				; up and down compensation gain adjust where z axis used to compensate x or y slope 
	btfss	PORTB,2
	goto	CALIBRATE			; deceleration calibration settings (VR3 and VR4)

UP_DN_RETURN ; location for UP_DN routine to start from, bypassing above jumper link checks

; delay for reading update
	movlw	D'5'				; 5ms per value (due to averaging over 8-values, multiply delay 8X for overall response)
	movwf	STORE3
DELAYX0
	call		DELAYms
	decfsz	STORE3,f
	goto	DELAYX0

; read X or Y  and Z. Compensate for up down slope and place into DECELERATE 

; measure X or Y
; AN3 for X or Y value
; bits 5 4 3
; ch3  0 1 1
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5

; Channel 3 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	XY_RUN			; value of XY output (ie = or above or below quiescent value)

; take XY value from horizontal XY quiescent 
	clrf		MORE_LESS	; more or less than flag. Begin with flag clear
	subwf	XY_HORIZ,w
	btfss	STATUS,C		; if negative, XY output is  increasing above quiescent
	bsf		MORE_LESS,0	; set flag when negative

; AN2 for Z value
; bits 5 4 3
; ch2  0 1 0
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5

; Channel 2 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
; take Z value from horizontal Z quiescent 
	subwf	Z_HORIZ,w
	btfss	STATUS,C		; if negative set result to 0
	clrw					; clear
	movwf	Z_RUN			; value below quiescent 

; get divisor value according to Z offset from the quiescent value

; set up non-linear piecewise gain table for the Z output compensation (non-linearity is due to Z output following a cosine curve and XY a sine curve) 
; piecewise gain steps are mostly in 5 degree steps. Overall gain can be adjusted using separate UP and Down multiplier values.
; gain of Z output is 4x compared to X/Y axis so a gains mentioned below are on top of the 4x already set

NXT0; 0-2.5 degrees
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'1'
	btfss	STATUS,C
	goto	NXT2_5
	movlw	D'30';	
	goto	STO_DIV

; Note: gains refered to below are extra to the X4 already set

NXT2_5;  2.5 to 5 degrees. Gain at 5 degrees overall required is x22 
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'2'
	btfss	STATUS,C
	goto	NXT5
	movlw	D'23'				; divisor value for Z output offset from quiescent for a gain of 5.5 (multiplier at 127) (ie 64 x 2))
	goto	STO_DIV

NXT5 ;5-10 degrees. Gain at 10 degrees overall required is x11
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'4'
	btfss	STATUS,C
	goto	NXT10
	movlw	D'47'				; divisor value for Z output offset from quiescent for a gain of 2.7  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT10; 10-15 degrees. Gain at 15 degrees overall required is x7.6
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'10'
	btfss	STATUS,C
	goto	NXT15
	movlw	D'67'				; divisor value for Z output offset from quiescent for a gain of 1.9  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT15; 15-20 degrees. Gain at 20 degrees overall required is x5.6
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'17'
	btfss	STATUS,C
	goto	NXT20
	movlw	D'91'				; divisor value for Z output offset from quiescent for a gain of 1.4  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT20; 20-25 degrees. Gain at 25 degrees overall required is x4.5
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'27'
	btfss	STATUS,C
	goto	NXT25
	movlw	D'113'				; divisor value for Z output offset from quiescent for a gain of 1.12  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT25; 25-30 degrees. Gain at 30 degrees overall required is x3.75
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'38'
	btfss	STATUS,C
	goto	NXT30
	movlw	D'137'				; divisor value for Z output offset from quiescent for a gain of 0.925  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT30; 30-35 degrees. Gain at 35 degrees overall required is x3.165
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'52'
	btfss	STATUS,C
	goto	NXT35
	movlw	D'160'				; divisor value for Z output offset from quiescent for a gain of 0.790  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT35; 35-40 degrees. Gain at 40 degrees overall required is x2.75
; is Z offset from quiescent value 
	movf	Z_RUN,w
	sublw	D'66'
	btfss	STATUS,C
	goto	NXT40
	movlw	D'185'				; divisor value for Z output offset from quiescent for a gain of 0.687  (assuming multiplier at 127 (ie 64 x 2))
	goto	STO_DIV

NXT40; 40 degrees onwards

	movlw	D'210' 				; use fixed divisor 

STO_DIV
	movwf	DIVISOR			; place w in divisor value
	call		CALC			; calculate 

; add or subtract to XY_RUN depending on MORE_LESS flag
	btfsc	MORE_LESS,0
	goto	COMP_MINUS

	movf	XY_RUN,w
	addwf	AARGB2,w		; revised  Z_RUN value increased 
	goto	COMP_XY

COMP_MINUS
	movf	AARGB2,w		; revised  Z_RUN value increased 
	subwf	XY_RUN,w

COMP_XY
	movwf	DECELERATE

;	goto	BY_AVG

; average out deceleration readings
; First move stored values along 

	movf	DECEL7,w
	movwf	DECEL8
	movf	DECEL6,w
	movwf	DECEL7
	movf	DECEL5,w
	movwf	DECEL6
	movf	DECEL4,w
	movwf	DECEL5
	movf	DECEL3,w
	movwf	DECEL4
	movf	DECEL2,w
	movwf	DECEL3
	movf	DECEL1,w
	movwf	DECEL2
	movf	DECELERATE,w
	movwf	DECEL1

; add then up into ADDN0,ADDN1
	clrf		ADDN0
	movf	DECEL1,w
	movwf	ADDN1
	movf	DECEL2,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	DECEL3,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	DECEL4,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	DECEL5,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	DECEL6,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	DECEL7,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f
	movf	DECEL8,w
	addwf	ADDN1,f
	btfsc	STATUS,C	; if over, add 1 to ADDN0
	incf		ADDN0,f

; divide by 8
;/2
	bcf		STATUS,C
	rrf		ADDN0,f
	rrf		ADDN1,f
;/4
	bcf		STATUS,C
	rrf		ADDN0,f
	rrf		ADDN1,f
;/8
	bcf		STATUS,C
	rrf		ADDN0,f
	rrf		ADDN1,w
	movwf	DECELERATE

BY_AVG
	movf	DECELERATE,w
	btfsc	STATUS,Z		; if zero use 1	
	movlw	D'1'
	movwf	DECELERATE	;  averaged value

; alternate output selection; average or not
; averaged output to PWM
	movwf	CCPR1L		; decelerate averaged value ms byte of PWM
; alternative current value (not averaged) output to PWM
;	movf	DECEL1,w		; decelerate current value	
;	movwf	CCPR1L		; ms byte of PWM

; bypass if in UP_DN calibration for gain
	btfsc	PORTB,1
	goto	MONITOR
	call		END_STOP_SIGNALLING	; stop flashing LEDs
	goto 	NORMAL 				;(normal running) 

MONITOR

	goto	BY_LOG				; remark out this instruction for logging 
; >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; log output variation using EEPROM as a memory from EEPROM3 to 255
	incf		LOGGER,w
	btfsc	STATUS,Z		; if increased past 255 to zero, then stop logging
	goto	BY_LOG
	incf		LOGGER,f		; increase logger address
	bsf 		STATUS,RP1	; select memory bank 2
	movwf 	EEADR			; indirect special function register
	bcf 		STATUS,RP1	; select memory bank 0
	movf	DECELERATE,w			; deceleration value
	call		EEWRITE		; store in EEPROM
; end logging
;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	
BY_LOG
; monitor averaged deceleration for emergency braking 
	movf	DECELERATE,w			; deceleration value
	subwf	UPPER_TRS,w			; upper threshold
	btfss	STATUS,C
	goto	BRAKES

; check for below threshold
	btfsc	BRAKE,1				; if brakes signalling on then check for lower threshold.
	goto	LOWER_THRESHOLD
	call		END_STOP_SIGNALLING
	goto 	NORMAL				 ;(normal running) 

LOWER_THRESHOLD
	movf	DECELERATE,w			; deceleration value
	subwf	LOWER_TRS,w			; lower threshold
	btfsc	STATUS,C
	call		END_STOP_SIGNALLING
	goto	NORMAL				 ;(normal running) 

BRAKES
	btfss	BRAKE,1				; if set then goto loop
	call		BEGIN_STOP_SIGNALLING	; start emergency brake signalling. Once started monitor for lower threshold for off
	goto 	NORMAL 				;(normal running) 

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

QUIESCENT			; read and store the quiescient X or Y and Z output when horizontal 

; delay
	movlw	D'250'
	movwf	STORE3
DELAYX
	call		DELAYms
	decfsz	STORE3,f
	goto	DELAYX

; measure quiescent (horizontal outputs)
; AN3 for X or Y value
; bits 5 4 3
; ch3  0 1 1
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5

; Channel 3 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	XY_HORIZ		; X or Y horizontal quiescent value

; AN2 for Z value
; bits 5 4 3
; ch2  0 1 0
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5

; Channel 2 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	Z_HORIZ		; Z horizontal quiescent value

; store X or Y level in EEPROM
	movlw	EEPROM0
	call 		EEREAD		; sets EEADR
	movf	XY_HORIZ,w
	call		EEWRITE
	
; store Z level in EEPROM
	movlw	EEPROM1
	call 		EEREAD		; sets EEADR
	movf	Z_HORIZ,w
	call		EEWRITE

; light LED1

	clrf		PORTA
	clrf		PORTA_STO		; store PORTA value

; wait for JP2 out	
LOOPXYZ
	call		DELAYms
	btfss	PORTB,5
	goto	LOOPXYZ
	call		DELAYms;
; LED1 off
	bsf		PORTA_STO,1	; portA value stored
	bsf		PORTA,1		; quiescent readings finished
	goto	NORMAL

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>	
UP_DN			; up and down compensation calibration. Adjust VR3 for up and down hill or slope tracking (0-30 degrees)

; AN6 VR3 re-tasked and temporarily sets compensation gain (multiply) (VR3 also for upper threshold adjustment)
; bits 5 4 3
; ch6  1 1 0
	bcf		ADCON0,3
	bsf		ADCON0,4
	bsf		ADCON0,5

; Channel 6 A/D value
	call		ACQUIRE_AD
	comf	ADRESH,w
	movwf	MULTIPLY

; Store value in EEPROM
	movlw	EEPROM2
	call 		EEREAD		; sets EEADR
	movf	MULTIPLY,w
	call		EEWRITE

; check if JP3 jumper for UP/DN compensation is in or out
	call		DELAYms
	btfsc	PORTB,1
	goto	NORMAL			; run normal program
	goto	UP_DN_RETURN		; run the normal program but without threshold checks and bypass the first set of jumper connection checks


;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALIBRATE			; calibrate deceleration thresholds using VR3 and VR4
; read and store thresholds and flash LEDs when deceleration is over upper threshold (when LEDs triggered on) and then off when below lower threshold.
; only monitor X or Y signal preventing Z compensation so gravity can be used as a calibration source

	clrf		BRAKE		; emergency brake flag, on or off. clear for off
; read threshold trim pots
CONT_CAL
	call 		DELAYms
; exit when jumper is out
	btfss	PORTB,2	; when high exit
	goto	READ_VR3_4
	call		END_STOP_SIGNALLING	; stop flashing LED and relay
	goto	UL			; begin running of program

READ_VR3_4	
; read VR3 and VR4 (upper and lower deceleration thresholds )

; AN6 VR3 UPPER threshold
; bits 5 4 3
; ch6  1 1 0
	bcf		ADCON0,3
	bsf		ADCON0,4
	bsf		ADCON0,5

; Channel 6 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	UPPER_TRS

; AN5 VR4 LOWER threshold
; bits 5 4 3
; ch3  1 0 1
	bsf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5

; Channel 5 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	LOWER_TRS

; read X or Y output

; bits 5 4 3
; ch3 0 1 1
	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5

; Channel 3 A/D value
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	DECELERATE	
; output to PWM
	movwf	CCPR1L				; ms byte of PWM
	
; monitor for thresholds

; monitor  
	movf	DECELERATE,w			; deceleration value
	subwf	UPPER_TRS,w			; upper threshold
	btfss	STATUS,C
	goto	BRAKE_CAL

; check for below threshold
	btfsc	BRAKE,1				; if brakes signalling on then check for lower threshold.
	goto	LOWER_THRESHOLD1
	call		END_STOP_SIGNALLING
	goto 	CONT_CAL

LOWER_THRESHOLD1
	movf	DECELERATE,w			; deceleration value
	subwf	LOWER_TRS,w			; lower threshold
	btfsc	STATUS,C
	call		END_STOP_SIGNALLING
	goto	CONT_CAL				 ;

BRAKE_CAL
	btfss	BRAKE,1				; if set then goto loop
	call		BEGIN_STOP_SIGNALLING	; start emergency brake signalling. Once started monitor for lower threshold for off
	goto 	CONT_CAL 				; 

; xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

; Subroutines

; emergency stop lights
BEGIN_STOP_SIGNALLING
; allow interrupts
	incf		BRAKE,f			; increase
	incf		BRAKE,f			; remark out if 2 'over upper thresholds' needed
	btfss	BRAKE,1		; when set started
	return
	clrf		DIVIDE			; interrupt divider	
	bsf		INTCON,TMR0IE	; timer 0 interrupt enable
	bcf		INTCON,TMR0IF	; clear TMR0 interrupt flag
; <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
; remark out as appropriate
; test ; relays off, LED on version
;	movlw	B'00000000'
; Normal: relays on, LED on
	movlw	B'10000001'		; RA1 for LED1, RA7 for Q1 and RA0 for Q2 all on
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	movwf	PORTA_STO		; store PORTA value
	movwf	PORTA
	bsf 		INTCON,GIE		; set global interrupt enable 
	return

END_STOP_SIGNALLING
	bcf 		INTCON,GIE		; clear global interrupt enable 
	movlw	B'00000010'		; RA1 for LED1, RA7 for Q1 and RA0 for Q2 all off
	movwf	PORTA_STO		; store PORTA value
	movwf	PORTA
	clrf		BRAKE			; clear when stopped
	return

; subroutine to wait for A/D conversion

ACQUIRE_AD
; wait for >20us
	movlw	D'40'
	movwf	CONVERSION
LOOP_CONV
	decfsz	CONVERSION,f	; decrease 
	goto	LOOP_CONV	
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return


; delay loop 

DELAYms
	movlw	D'23'		; delay value
DELAYY
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'70'
DELDSP
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8		; decrease till STORE1 is zero
	return


; subroutine to read EEPROM memory 
EEREAD
	bcf		STATUS,RP0	; select bank 0
	bsf 		STATUS,RP1	; select memory bank 2
	movwf 	EEADR			; indirect special function register
	bsf 		STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD		; read EEPROM
	bcf 		STATUS,RP0	; select memory bank 2
	movf	EEDATA,w		; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM interrupt not enabled
EEWRITE
	bcf 		STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 		STATUS,RP0	; select memory bank 3
WR4	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR4			; not written yet
	bcf		INTCON,GIE		; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'			; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'			; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit

WRITE1
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE1			; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 		STATUS,RP0	; select memory bank 0
	return					; value written
 
; multiply and divide for slope compensation tracking

CALC
	clrf		AARGB2		;clear ready if divisor=0
	clrf		AARGB0		; clear
; get multiplier
	movf	MULTIPLY,w		; initially set at D'92'
	movwf	AARGB1
; multiply MULTIPLY by 2
	bcf		STATUS,C
	rlf		AARGB1,f
	rlf		AARGB0,f
MULT
	movf	DIVISOR,w	; divisor
; check if zero
	btfsc	STATUS,Z
	return					; bypass calc with AARGB2=0

DO_CALC
	movf	Z_RUN,w		
; check if zero
	btfsc	STATUS,Z
	return					; bypass calc with AARGB2=0
	movwf	BARGB1
	clrf		BARGB0
	call		FXM1616U	; multiply 

; shift answer
	movf	AARGB2,w	; second ls byte of result
	movwf	STO1
	movf	AARGB3,w
	movwf	AARGB2
	movf	AARGB1,w
	movwf	AARGB0
	movf	STO1,w
	movwf	AARGB1
	
; load denominator

	movf	DIVISOR,w	; divisor
	movwf	BARGB1
	clrf		BARGB0
	call		FXD2416U	; divide 
; result in AARGB1, AARGB2
	return


; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U   
	       CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A 
		RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA 
                ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LA 
		RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8 
               ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L8  
		RLF             AARGB1, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B 
		RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB  
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LB
	        RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16 
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L16 
	        RLF             AARGB2, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C  
		RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC  
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LC 
		RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L	
		RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U 
	        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A 
		RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B 
	        RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   H'00'
BLUM1616NAP  
		 BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP 
		BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616 
	        RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA   
	        RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616 
		RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA  
		RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN

 
	end
